import os
import subprocess
from datetime import timedelta
from flask import Flask, render_template, request, redirect, url_for, flash, send_from_directory, abort
from flask_sqlalchemy import SQLAlchemy
from flask_login import LoginManager, login_user, logout_user, login_required, current_user, UserMixin
from flask_wtf import FlaskForm
from flask_wtf.csrf import CSRFProtect
from flask_wtf.file import FileField, FileAllowed
from wtforms import StringField, TextAreaField, PasswordField, SubmitField, BooleanField
from wtforms.validators import DataRequired, Email, NumberRange
from werkzeug.security import generate_password_hash, check_password_hash
from werkzeug.utils import secure_filename
from PIL import Image

# ----------------------------
# إعدادات التطبيق
# ----------------------------

app = Flask(__name__)
app.config.from_object('config.Config')

# تفعيل حماية CSRF (مهم للأمان)
csrf = CSRFProtect(app)

# إنشاء مجلدات الرفع إذا لم تكن موجودة
os.makedirs(os.path.join(app.config['UPLOAD_FOLDER'], 'products'), exist_ok=True)
os.makedirs(os.path.join(app.config['UPLOAD_FOLDER'], 'brands'), exist_ok=True)
os.makedirs(os.path.join(app.config['UPLOAD_FOLDER'], 'hero'), exist_ok=True)
os.makedirs(os.path.join(app.config['UPLOAD_FOLDER'], 'videos'), exist_ok=True)
os.makedirs(os.path.join(app.static_folder, 'uploads'), exist_ok=True)

# ----------------------------
# قاعدة البيانات والنماذج
# ----------------------------

db = SQLAlchemy(app)

class User(UserMixin, db.Model):
    id = db.Column(db.Integer, primary_key=True)
    username = db.Column(db.String(80), unique=True, nullable=False)
    password_hash = db.Column(db.String(128), nullable=False)

    def set_password(self, password):
        self.password_hash = generate_password_hash(password)

    def check_password(self, password):
        return check_password_hash(self.password_hash, password)

class Brand(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(100), nullable=False)
    logo = db.Column(db.String(200))
    products = db.relationship('Product', backref='brand', lazy=True, cascade='all, delete-orphan')

class Product(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(100), nullable=False)
    description = db.Column(db.Text)
    image = db.Column(db.String(200))
    brand_id = db.Column(db.Integer, db.ForeignKey('brand.id'), nullable=False)

class Message(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(100), nullable=False)
    subject = db.Column(db.String(200))
    body = db.Column(db.Text, nullable=False)
    read = db.Column(db.Boolean, default=False)

class HeroImage(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    filename = db.Column(db.String(200), nullable=False)
    caption = db.Column(db.String(200))
    active = db.Column(db.Boolean, default=True)

class GalleryImage(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    filename = db.Column(db.String(200), nullable=False)
    caption = db.Column(db.String(200))
    active = db.Column(db.Boolean, default=True)

class Video(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    filename = db.Column(db.String(200), nullable=False)
    title = db.Column(db.String(200), nullable=False)
    active = db.Column(db.Boolean, default=True)

# ----------------------------
# إدارة تسجيل الدخول
# ----------------------------

login_manager = LoginManager()
login_manager.init_app(app)
login_manager.login_view = 'admin_login'
login_manager.refresh_view = 'admin_login'
login_manager.needs_refresh_message = 'يرجى تسجيل الدخول مجددًا لأسباب أمنية.'
login_manager.needs_refresh_message_category = 'info'

@login_manager.user_loader
def load_user(user_id):
    return User.query.get(int(user_id))

# ----------------------------
# نماذج النماذج (Forms)
# ----------------------------

class LoginForm(FlaskForm):
    username = StringField('اسم المستخدم', validators=[DataRequired()])
    password = PasswordField('كلمة المرور', validators=[DataRequired()])
    submit = SubmitField('تسجيل الدخول')

class ContactForm(FlaskForm):
    name = StringField('الاسم', validators=[DataRequired()])
    subject = StringField('الموضوع')
    body = TextAreaField('الرسالة', validators=[DataRequired()])
    submit = SubmitField('إرسال')

class ProductForm(FlaskForm):
    name = StringField('اسم المنتج', validators=[DataRequired()])
    description = TextAreaField('الوصف')
    image = FileField('صورة المنتج', validators=[FileAllowed(['png', 'jpg', 'jpeg', 'webp'], 'صور فقط!')])
    brand_id = StringField('معرف العلامة التجارية', validators=[DataRequired(), NumberRange(min=1, message="يجب أن يكون معرف العلامة رقمًا صحيحًا.")])
    submit = SubmitField('حفظ')

class BrandForm(FlaskForm):
    name = StringField('اسم العلامة', validators=[DataRequired()])
    logo = FileField('شعار العلامة', validators=[FileAllowed(['png', 'jpg', 'jpeg', 'webp'], 'صور فقط!')])
    submit = SubmitField('حفظ')

class AddHeroImageForm(FlaskForm):
    image = FileField('صورة الهيدر', validators=[DataRequired(), FileAllowed(['png', 'jpg', 'jpeg', 'webp'], 'صور فقط!')])
    caption = StringField('التعليق (اختياري)')
    active = BooleanField('مفعلة في العرض')
    submit = SubmitField('حفظ')

class EditHeroImageForm(FlaskForm):
    image = FileField('صورة الهيدر', validators=[FileAllowed(['png', 'jpg', 'jpeg', 'webp'], 'صور فقط!')])
    caption = StringField('التعليق (اختياري)')
    active = BooleanField('مفعلة في العرض')
    submit = SubmitField('تحديث')

class GalleryImageForm(FlaskForm):
    image = FileField('صورة المعرض', validators=[FileAllowed(['png', 'jpg', 'jpeg', 'webp'], 'صور فقط!')])
    caption = StringField('التعليق (اختياري)')
    active = BooleanField('مفعلة في العرض')
    submit = SubmitField('حفظ')

class VideoForm(FlaskForm):
    title = StringField('عنوان الفيديو', validators=[DataRequired()])
    video = FileField('ملف الفيديو', validators=[FileAllowed(['mp4', 'mov', 'avi', 'mkv'], 'فيديو فقط!')])
    active = BooleanField('مفعل')
    submit = SubmitField('حفظ')

# ----------------------------
# وظائف مساعدة
# ----------------------------

def save_image(file, folder):
    if not file or not file.filename:
        return None
    filename = secure_filename(file.filename)
    if not filename:
        return None
    # منع أي محاولة للخروج من المجلد
    if '..' in filename or filename.startswith('/'):
        return None
    path = os.path.join(app.config['UPLOAD_FOLDER'], folder, filename)
    counter = 1
    while os.path.exists(path):
        name, ext = os.path.splitext(filename)
        filename = f"{name}_{counter}{ext}"
        path = os.path.join(app.config['UPLOAD_FOLDER'], folder, filename)
        counter += 1
    file.save(path)
    try:
        img = Image.open(path)
        img.thumbnail((800, 800))
        img.save(path, optimize=True, quality=75)
    except Exception as e:
        print(f"Warning: Could not optimize image: {e}")
    return os.path.join('uploads', folder, filename).replace('\\', '/')

def save_video(file, folder):
    if not file or not file.filename:
        return None
    filename = secure_filename(file.filename)
    if not filename:
        return None
    if '..' in filename or filename.startswith('/'):
        return None

    _, ext = os.path.splitext(filename)
    ext = ext.lower()
    if ext not in app.config.get('ALLOWED_VIDEO_EXTENSIONS', {'mp4', 'mov', 'avi', 'mkv'}):
        return None

    path = os.path.join(app.config['UPLOAD_FOLDER'], folder, filename)
    file.save(path)

    compressed_filename = filename.replace(ext, f"_compressed{ext}")
    compressed_path = os.path.join(app.config['UPLOAD_FOLDER'], folder, compressed_filename)

    try:
        subprocess.run(
            ["ffmpeg", "-i", path, "-vcodec", "libx264", "-crf", "28", "-y", compressed_path],
            check=True, stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL
        )
        os.remove(path)
        return compressed_filename
    except Exception as e:
        print("Video compression failed:", e)
        if os.path.exists(path):
            os.remove(path)
        return None

# ----------------------------
# المسارات (Routes)
# ----------------------------

@app.route('/uploads/<path:filename>')
def uploaded_file(filename):
    upload_dir = os.path.join(app.static_folder, 'uploads')
    # منع Directory Traversal
    if '..' in filename or filename.startswith('/'):
        abort(404)
    full_path = os.path.join(upload_dir, filename)
    if not os.path.exists(full_path):
        abort(404)
    return send_from_directory(upload_dir, filename)

@app.route('/')
def index():
    hero_images = HeroImage.query.filter_by(active=True).all()
    gallery_images = GalleryImage.query.filter_by(active=True).all()
    videos = Video.query.filter_by(active=True).all()
    return render_template('index.html', hero_images=hero_images, gallery_images=gallery_images, videos=videos)

@app.route('/agents')
def agents():
    return render_template('agents.html')

@app.route('/products')
def products():
    brand_id = request.args.get('brand')
    brands = Brand.query.all()
    if brand_id and brand_id.isdigit():
        products = Product.query.filter_by(brand_id=int(brand_id)).all()
    else:
        products = Product.query.all()
    return render_template('products.html', brands=brands, products=products, selected_brand=brand_id)

@app.route('/about')
def about():
    return render_template('about.html')

@app.route('/contact', methods=['GET', 'POST'])
def contact():
    form = ContactForm()
    if form.validate_on_submit():
        msg = Message(
            name=form.name.data,
            subject=form.subject.data,
            body=form.body.data
        )
        db.session.add(msg)
        db.session.commit()
        flash('تم إرسال رسالتك بنجاح!', 'success')
        return redirect(url_for('contact'))
    return render_template('contact.html', form=form)

# --- لوحة التحكم ---

@app.route('/admin/login', methods=['GET', 'POST'])
def admin_login():
    if current_user.is_authenticated:
        return redirect(url_for('admin_dashboard'))
    form = LoginForm()
    if form.validate_on_submit():
        user = User.query.filter_by(username=form.username.data).first()
        if user and user.check_password(form.password.data):
            login_user(user, remember=False)
            return redirect(url_for('admin_dashboard'))
        else:
            flash('اسم المستخدم أو كلمة المرور غير صحيحة.', 'danger')
    return render_template('admin/login.html', form=form)

@app.route('/admin/logout')
@login_required
def admin_logout():
    logout_user()
    flash('تم تسجيل الخروج.', 'info')
    return redirect(url_for('index'))

@app.route('/admin')
@login_required
def admin_dashboard():
    return render_template('admin/dashboard.html')

# --- إدارة صور الهيدر ---
@app.route('/admin/hero')
@login_required
def manage_hero():
    images = HeroImage.query.all()
    return render_template('admin/manage_hero.html', images=images)

@app.route('/admin/hero/add', methods=['GET', 'POST'])
@login_required
def add_hero():
    form = AddHeroImageForm()
    if form.validate_on_submit():
        image_path = save_image(form.image.data, 'hero')
        if not image_path:
            flash('فشل رفع الصورة. تأكد من الصيغة والحجم.', 'danger')
            return render_template('admin/add_hero.html', form=form)
        hero = HeroImage(
            filename=image_path,
            caption=form.caption.data,
            active=form.active.data
        )
        db.session.add(hero)
        db.session.commit()
        flash('تم إضافة صورة الهيدر!', 'success')
        return redirect(url_for('manage_hero'))
    return render_template('admin/add_hero.html', form=form)

@app.route('/admin/hero/edit/<int:id>', methods=['GET', 'POST'])
@login_required
def edit_hero(id):
    hero = HeroImage.query.get_or_404(id)
    form = EditHeroImageForm(obj=hero)
    if form.validate_on_submit():
        if form.image.data and form.image.data.filename:
            if hero.filename:
                old_path = os.path.join(app.static_folder, hero.filename)
                if os.path.exists(old_path):
                    os.remove(old_path)
            new_path = save_image(form.image.data, 'hero')
            if new_path:
                hero.filename = new_path
        hero.caption = form.caption.data
        hero.active = form.active.data
        db.session.commit()
        flash('تم تحديث الصورة!', 'success')
        return redirect(url_for('manage_hero'))
    return render_template('admin/edit_hero.html', form=form, hero=hero)

@app.route('/admin/hero/delete/<int:id>', methods=['POST'])
@login_required
def delete_hero(id):
    hero = HeroImage.query.get_or_404(id)
    if hero.filename:
        file_path = os.path.join(app.static_folder, hero.filename)
        if os.path.exists(file_path):
            os.remove(file_path)
    db.session.delete(hero)
    db.session.commit()
    flash('تم حذف الصورة نهائيًا.', 'info')
    return redirect(url_for('manage_hero'))

# --- إدارة المعرض ---
@app.route('/admin/gallery')
@login_required
def manage_gallery():
    images = GalleryImage.query.all()
    return render_template('admin/manage_gallery.html', images=images)

@app.route('/admin/gallery/add', methods=['GET', 'POST'])
@login_required
def add_gallery():
    form = GalleryImageForm()
    if form.validate_on_submit():
        image_path = save_image(form.image.data, 'gallery')
        if not image_path:
            flash('فشل رفع الصورة.', 'danger')
            return render_template('admin/add_gallery.html', form=form)
        gallery = GalleryImage(
            filename=image_path,
            caption=form.caption.data,
            active=form.active.data
        )
        db.session.add(gallery)
        db.session.commit()
        flash('تم إضافة صورة المعرض!', 'success')
        return redirect(url_for('manage_gallery'))
    return render_template('admin/add_gallery.html', form=form)

@app.route('/admin/gallery/edit/<int:id>', methods=['GET', 'POST'])
@login_required
def edit_gallery(id):
    gallery = GalleryImage.query.get_or_404(id)
    form = GalleryImageForm(obj=gallery)
    if form.validate_on_submit():
        if form.image.data and form.image.data.filename:
            if gallery.filename:
                old_path = os.path.join(app.static_folder, gallery.filename)
                if os.path.exists(old_path):
                    os.remove(old_path)
            new_path = save_image(form.image.data, 'gallery')
            if new_path:
                gallery.filename = new_path
        gallery.caption = form.caption.data
        gallery.active = form.active.data
        db.session.commit()
        flash('تم تحديث الصورة!', 'success')
        return redirect(url_for('manage_gallery'))
    return render_template('admin/edit_gallery.html', form=form, gallery=gallery)

@app.route('/admin/gallery/delete/<int:id>', methods=['POST'])
@login_required
def delete_gallery(id):
    gallery = GalleryImage.query.get_or_404(id)
    if gallery.filename:
        file_path = os.path.join(app.static_folder, gallery.filename)
        if os.path.exists(file_path):
            os.remove(file_path)
    db.session.delete(gallery)
    db.session.commit()
    flash('تم حذف الصورة نهائيًا.', 'info')
    return redirect(url_for('manage_gallery'))

# --- إدارة المنتجات ---
@app.route('/admin/products')
@login_required
def manage_products():
    products = Product.query.all()
    brands = Brand.query.all()
    return render_template('admin/manage_products.html', products=products, brands=brands)

@app.route('/admin/products/add', methods=['GET', 'POST'])
@login_required
def add_product():
    form = ProductForm()
    brands = Brand.query.all()
    if form.validate_on_submit():
        try:
            brand_id = int(form.brand_id.data)
        except (ValueError, TypeError):
            flash('معرف العلامة غير صالح.', 'danger')
            return render_template('admin/add_product.html', form=form, brands=brands)
        if not Brand.query.get(brand_id):
            flash('العلامة التجارية غير موجودة.', 'danger')
            return render_template('admin/add_product.html', form=form, brands=brands)

        image_path = save_image(form.image.data, 'products') if form.image.data else None
        product = Product(
            name=form.name.data,
            description=form.description.data,
            image=image_path,
            brand_id=brand_id
        )
        db.session.add(product)
        db.session.commit()
        flash('تم إضافة المنتج بنجاح!', 'success')
        return redirect(url_for('manage_products'))
    return render_template('admin/add_product.html', form=form, brands=brands)

@app.route('/admin/products/edit/<int:id>', methods=['GET', 'POST'])
@login_required
def edit_product(id):
    product = Product.query.get_or_404(id)
    form = ProductForm(obj=product)
    brands = Brand.query.all()

    if form.validate_on_submit():
        try:
            brand_id = int(form.brand_id.data)
        except (ValueError, TypeError):
            flash('معرف العلامة غير صالح.', 'danger')
            return render_template('admin/edit_product.html', form=form, product=product, brands=brands)
        if not Brand.query.get(brand_id):
            flash('العلامة التجارية غير موجودة.', 'danger')
            return render_template('admin/edit_product.html', form=form, product=product, brands=brands)

        product.name = form.name.data
        product.description = form.description.data
        product.brand_id = brand_id

        if form.image.data and form.image.data.filename:
            if product.image:
                old_path = os.path.join(app.static_folder, product.image)
                if os.path.exists(old_path):
                    os.remove(old_path)
            product.image = save_image(form.image.data, 'products')

        db.session.commit()
        flash('تم تحديث المنتج!', 'success')
        return redirect(url_for('manage_products'))

    return render_template('admin/edit_product.html', form=form, product=product, brands=brands)

@app.route('/admin/products/delete/<int:id>', methods=['POST'])
@login_required
def delete_product(id):
    product = Product.query.get_or_404(id)
    if product.image:
        file_path = os.path.join(app.static_folder, product.image)
        if os.path.exists(file_path):
            os.remove(file_path)
    db.session.delete(product)
    db.session.commit()
    flash('تم حذف المنتج نهائيًا.', 'info')
    return redirect(url_for('manage_products'))

# --- إدارة العلامات التجارية ---
@app.route('/admin/brands')
@login_required
def manage_brands():
    brands = Brand.query.all()
    return render_template('admin/manage_brands.html', brands=brands)

@app.route('/admin/brands/add', methods=['GET', 'POST'])
@login_required
def add_brand():
    form = BrandForm()
    if form.validate_on_submit():
        logo_path = save_image(form.logo.data, 'brands') if form.logo.data else None
        brand = Brand(name=form.name.data, logo=logo_path)
        db.session.add(brand)
        db.session.commit()
        flash('تمت إضافة العلامة التجارية!', 'success')
        return redirect(url_for('manage_brands'))
    return render_template('admin/add_brand.html', form=form)

@app.route('/admin/brands/edit/<int:id>', methods=['GET', 'POST'])
@login_required
def edit_brand(id):
    brand = Brand.query.get_or_404(id)
    form = BrandForm(obj=brand)
    if form.validate_on_submit():
        brand.name = form.name.data
        if form.logo.data and form.logo.data.filename:
            if brand.logo:
                old_path = os.path.join(app.static_folder, brand.logo)
                if os.path.exists(old_path):
                    os.remove(old_path)
            brand.logo = save_image(form.logo.data, 'brands')
        db.session.commit()
        flash('تم تحديث العلامة!', 'success')
        return redirect(url_for('manage_brands'))
    return render_template('admin/edit_brand.html', form=form, brand=brand)

@app.route('/admin/brands/delete/<int:id>', methods=['POST'])
@login_required
def delete_brand(id):
    brand = Brand.query.get_or_404(id)
    if brand.logo:
        file_path = os.path.join(app.static_folder, brand.logo)
        if os.path.exists(file_path):
            os.remove(file_path)
    db.session.delete(brand)
    db.session.commit()
    flash('تم حذف العلامة التجارية نهائيًا.', 'info')
    return redirect(url_for('manage_brands'))

# --- إدارة الرسائل ---
@app.route('/admin/messages')
@login_required
def manage_messages():
    messages = Message.query.order_by(Message.id.desc()).all()
    return render_template('admin/messages.html', messages=messages)

@app.route('/admin/messages/read/<int:id>')
@login_required
def mark_message_read(id):
    msg = Message.query.get_or_404(id)
    msg.read = True
    db.session.commit()
    return redirect(url_for('manage_messages'))

@app.route('/admin/messages/delete/<int:id>', methods=['POST'])
@login_required
def delete_message(id):
    msg = Message.query.get_or_404(id)
    db.session.delete(msg)
    db.session.commit()
    flash('تم حذف الرسالة.', 'info')
    return redirect(url_for('manage_messages'))

# --- إدارة الفيديوهات ---
@app.route('/admin/videos')
@login_required
def manage_videos():
    videos = Video.query.all()
    return render_template('admin/manage_videos.html', videos=videos)

@app.route('/admin/videos/add', methods=['GET', 'POST'])
@login_required
def add_video():
    form = VideoForm()
    if form.validate_on_submit():
        video_path = save_video(form.video.data, 'videos')
        if not video_path:
            flash("فشل رفع الفيديو أو ضغطه. تأكد من الصيغة والحجم.", "danger")
            return redirect(url_for('add_video'))
        video = Video(
            title=form.title.data,
            filename=video_path,
            active=form.active.data
        )
        db.session.add(video)
        db.session.commit()
        flash('✅ تم إضافة الفيديو!', 'success')
        return redirect(url_for('manage_videos'))
    return render_template('admin/add_video.html', form=form)

@app.route('/admin/videos/edit/<int:id>', methods=['GET', 'POST'])
@login_required
def edit_video(id):
    video = Video.query.get_or_404(id)
    form = VideoForm(obj=video)
    if form.validate_on_submit():
        video.title = form.title.data
        video.active = form.active.data
        if form.video.data and form.video.data.filename:
            old_path = os.path.join(app.config['UPLOAD_FOLDER'], 'videos', video.filename)
            if os.path.exists(old_path):
                os.remove(old_path)
            new_path = save_video(form.video.data, 'videos')
            if new_path:
                video.filename = new_path
        db.session.commit()
        flash('✅ تم تحديث الفيديو!', 'success')
        return redirect(url_for('manage_videos'))
    return render_template('admin/edit_video.html', form=form, video=video)

@app.route('/admin/videos/delete/<int:id>', methods=['POST'])
@login_required
def delete_video(id):
    video = Video.query.get_or_404(id)
    if video.filename:
        file_path = os.path.join(app.config['UPLOAD_FOLDER'], 'videos', video.filename)
        if os.path.exists(file_path):
            os.remove(file_path)
    db.session.delete(video)
    db.session.commit()
    flash('🗑️ تم حذف الفيديو نهائيًا!', 'info')
    return redirect(url_for('manage_videos'))

# ----------------------------
# تشغيل التطبيق
# ----------------------------

if __name__ == '__main__':
    with app.app_context():
        db.create_all()
        # إنشاء مستخدم إداري افتراضي (فقط في وضع التطوير)
        if not User.query.filter_by(username='admin').first():
            admin_pass = os.environ.get('ADMIN_PASSWORD', 'ChangeMeInProduction!')
            admin = User(username='admin')
            admin.set_password(admin_pass)
            db.session.add(admin)
            db.session.commit()
            print("✅ تم إنشاء حساب إداري افتراضي: admin / (من متغير ADMIN_PASSWORD أو 'ChangeMeInProduction!')")
    # لا تستخدم debug=True في الإنتاج
    app.run(debug=os.environ.get('FLASK_ENV') == 'development')